/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"
#include "py32f092xx_ll_Start_Kit.h"

/* Private define ------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private user code ---------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void APP_SystemClockConfig(void);
static void APP_CompInit(void);
static void APP_EnterStop0(void);
static void APP_CompRccInit(void);

/**
  * @brief  Main program.
  * @param  None
  * @retval int
  */
int main(void)
{
  /* Enable SYSCFG and PWR clock */
  LL_APB1_GRP2_EnableClock(LL_APB1_GRP2_PERIPH_SYSCFG);
  LL_APB1_GRP1_EnableClock(LL_APB1_GRP1_PERIPH_PWR);

  /* Configure system clock */
  APP_SystemClockConfig();

  /* Initialize LED */
  BSP_LED_Init(LED_GREEN);
  BSP_LED_On(LED_GREEN);

  /* Initialize BUTTON */
  BSP_PB_Init(BUTTON_KEY,BUTTON_MODE_GPIO);

  /* Wait Button */
  while(BSP_PB_GetState(BUTTON_KEY) == 0)
  {
  }
  BSP_LED_Off(LED_GREEN);

  /* COMP Rcc Init */
  APP_CompRccInit();
  
  /* Initialize COMP1 */
  APP_CompInit();

  /* Enter Stop Mode */
  APP_EnterStop0();

  while (1)
  {
    BSP_LED_Toggle(LED_GREEN);

    LL_mDelay(200);
  }
}

/**
  * @brief  Initialize COMP1
  * @param  None
  * @retval None
  */
static void APP_CompInit(void)
{
  /* Enable GPIOD clock */
  LL_IOP_GRP1_EnableClock(LL_IOP_GRP1_PERIPH_GPIOD);

  /* Configure PD0 pin in analog mode */
  LL_GPIO_SetPinMode(GPIOD, LL_GPIO_PIN_0, LL_GPIO_MODE_ANALOG);
  LL_SYSCFG_EnableGPIOAnalog2(LL_SYSCFG_GPIO_PORTD, LL_SYSCFG_GPIO_PIN_0);

  /* Enable COMP1 clock */
  LL_APB1_GRP2_EnableClock(LL_APB1_GRP2_PERIPH_COMP1);

  /* Select VCC as VREFCMP reference source */
  LL_COMP_SetVrefCmpSource(LL_COMP_VREFCMP_SOURCE_VCC);

  /* Set VREFCMP divider : VREFCMP=1/2 VCC */
  LL_COMP_SetVrefCmpDivider(LL_COMP_VREFCMP_DIV_32_64VREFCMP);

  /* Enable Vrefcmp source */
  LL_COMP_EnableVrefCmp();

  /* Plus Select PD0 */
  LL_COMP_SetInputPlus(COMP1, LL_COMP_INPUT_PLUS_IO3);
  
  /* Minus Select VrefCmp */
  LL_COMP_SetInputMinus(COMP1, LL_COMP_INPUT_MINUS_IO3);

  /* Set comparator instance low power mode */
  LL_COMP_SetPowerMode(COMP1, LL_COMP_POWERMODE_MEDIUMSPEED);

  /* Disable WINDOW mode */
  LL_COMP_SetCommonWindowMode(__LL_COMP_COMMON_INSTANCE(COMP1), LL_COMP_WINDOWMODE_DISABLE);

  /* Enable Rising Trigger */
  LL_EXTI_EnableRisingTrig(LL_EXTI_LINE_17);

  /* Enable Interrupt */
  LL_EXTI_EnableIT(LL_EXTI_LINE_17);

  NVIC_SetPriority(ADC_COMP_IRQn, 0);
  NVIC_EnableIRQ(ADC_COMP_IRQn);

  /* Enable comp1 */
  LL_COMP_Enable(COMP1);

  __IO uint32_t wait_loop_index = 0;
  wait_loop_index = ((LL_COMP_DELAY_STARTUP_US / 10UL) * (SystemCoreClock / (100000UL * 2UL)));
  while(wait_loop_index != 0UL)
  {
    wait_loop_index--;
  }
}

/**
  * @brief  Comp Clock Init
  * @param  None
  * @retval None
  */
static void APP_CompRccInit()
{
  /* Enable access to the backup domain */
  LL_PWR_EnableBkUpAccess();
  
  /* Enable LSI */
  LL_RCC_LSI_Enable();
 
  /* Wait LSI Ready */
  while(LL_RCC_LSI_IsReady() != 1)
  {
  }
 
  /* Set LSC Source LSI */
  LL_RCC_LSC_SetSource(LL_RCC_LSC_CLKSOURCE_LSI);

  /* Set Comp1 Clock Source LSC */
  LL_RCC_SetCOMPClockSource(LL_RCC_COMP1_CLKSOURCE_LSC);
}

/**
  * @brief  Enter Stop
  * @param  None
  * @retval None
  */
static void APP_EnterStop0(void)
{
  /* Enable PWR Clock */
  LL_APB1_GRP1_EnableClock(LL_APB1_GRP1_PERIPH_PWR);

  LL_PWR_SetStopWakeUpFlashDelay(LL_PWR_STOPWAKEUP_FLASH_DELAY_5US);

  /* Set Low-Power mode */
  LL_PWR_SetPowerMode(LL_PWR_MODE_STOP0);
  
  /* STOP mode with DLPR ON */
  LL_PWR_SetVoltageRegulatorMode(LL_PWR_VOLTAGE_REGULATOR_MODE_DLPR);

  /* Voltage Regulator selects high mode */
  LL_PWR_SetDlprVoltageMode(LL_PWR_DLPR_VOLTAGE_HIGH);

  /* Enter DeepSleep mode */
  LL_LPM_EnableDeepSleep();

  /* Waiting for interrupt instructions */
  __WFI();

  LL_LPM_EnableSleep();
}

/**
  * @brief  Configure system clock
  * @param  None
  * @retval None
  */
static void APP_SystemClockConfig(void)
{
  /* Enable HSI */
  LL_RCC_HSI_Enable();
  while(LL_RCC_HSI_IsReady() != 1)
  {
  }

  /* Set AHB prescaler: HCLK = SYSCLK */
  LL_RCC_SetAHBPrescaler(LL_RCC_SYSCLK_DIV_1);

  /* Select HSISYS as system clock source */
  LL_RCC_SetSysClkSource(LL_RCC_SYS_CLKSOURCE_HSISYS);
  while(LL_RCC_GetSysClkSource() != LL_RCC_SYS_CLKSOURCE_STATUS_HSISYS)
  {
  }

  /* Set APB prescaler: PCLK = HCLK */
  LL_RCC_SetAPB1Prescaler(LL_RCC_APB1_DIV_1);
  LL_Init1msTick(8000000);

  /* Update the SystemCoreClock global variable(which can be updated also through SystemCoreClockUpdate function) */
  LL_SetSystemCoreClock(8000000);
}

/**
  * @brief  Error executing function.
  * @param  None
  * @retval None
  */
void APP_ErrorHandler(void)
{
  while (1)
  {
  }
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* Users can add their own printing information as needed,
     for example: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* Infinite loop */
  while (1)
  {
  }
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
